/* ...............................................................

	WindowColors
	Copyright 1997-8 Steve Klingsporn <moofie@pobox.com>
	Based on WindowShade by Marco Nelissen <marcone@xs4all.nl>
	
		File:	WindowColorsApplication.cpp
	
	Contains:	Implementation of a BApplication subclass.
	
	   Notes:	None.
	   
   ............................................................... */

#ifndef _WINDOW_COLORS_APPLICATION_H
#include "WindowColorsApplication.h"
#endif


/* ...............................................................	
	Global application pointer (with the right type)
   ............................................................... */

WindowColorsApplication	*window_colors_app = NULL;


/* ...............................................................	
	String Constants (that probably should be in resources)
   ............................................................... */
   
const char *APPLICATION_SIGNATURE 		= "application/x-vnd.moofie" \
										  "-WindowColors";
const char *SETTINGS_FILE_NAME 	  		= "WindowColors_settings";
const char *PREFERRED_SET_ENTRY			= "preferred_set";

const char *BOOTSCRIPT_ARGUMENT			= "-Bootscript";
const char *USER_BOOTSCRIPT_ARGUMENT	= "-UserBootscript";
const char *RESET_TO_DEFAULTS_ARGUMENT  = "-defaults";
const char *UPDATE_COLORS_ARGUMENT		= "-update";
		
const char *CANNOT_BE_USED_MESSAGE		= "\nThis version of " \
										  "WindowColors cannot be " \
										  "used with your version " \
										  "of BeOS.\n\nSee " \
										  "http://www.pobox.com/" \
										  "~moofie/beos/WindowColors" \
										  " for a version that\nis " \
										  "designed for your version" \
										  " of BeOS.\n";
										  
										  
/* ...............................................................	
	main
	Creates the (global) application object, runs it, and cleans
	up after we're done, returning B_OK.
   ............................................................... */

int main()
{
	WindowColorsApplication application;
	window_colors_app = &application;
	application.Run();
	window_colors_app = NULL;
	return B_OK;
}


/* ...............................................................	
	WindowColorsApplication Constructor
	Initializes the settings message to NULL.
   ............................................................... */

WindowColorsApplication::WindowColorsApplication()
					    :BApplication(APPLICATION_SIGNATURE),
					    _settings(NULL),
					    _window(NULL)
{
}


/* ...............................................................	
	WindowColorsApplication::ArgvReceived()
	Called when our application receives arguments from the
	command-line.  Parses the known argument types and sets the
	startup action based on them.
   ............................................................... */

void WindowColorsApplication::ArgvReceived(int32 argc, char **argv)
{
	_startupAction = NORMAL_STARTUP_ACTION;
	
	/* We'll only take one additional argument to the application
	   name on the command line. */   
	if (argc == 2)
	{
		if (strcmp(argv[1], BOOTSCRIPT_ARGUMENT) == 0)
			_startupAction = BOOTSCRIPT_ACTION;
			
		else if (strcmp(argv[1], USER_BOOTSCRIPT_ARGUMENT) == 0)
			_startupAction = USER_BOOTSCRIPT_ACTION;
			
		else if (strcmp(argv[1], RESET_TO_DEFAULTS_ARGUMENT) == 0)
			_startupAction = RESET_TO_DEFAULTS_ACTION;
			
		else if (strcmp(argv[1], UPDATE_COLORS_ARGUMENT) == 0)
			_startupAction = UPDATE_WINDOW_COLORS_ACTION;
	}
}


/* ...............................................................	
	WindowColorsApplication::ReadyToRun()
	Called when the application is just about ready to start
	performing.  We initialize the WindowColors API, posting an
	alert and quitting if we fail (like the app_server is the
	wrong version).  We load our settings file (a BMessage
	flattened to a file), and display a window if we should,
	otherwise performing the desired startup action.
	
	This is really contorted.  I'll clean it up for the next
	release.
   ............................................................... */

void WindowColorsApplication::ReadyToRun()
{
	/* Initialize the WindowColors API.  If this fails,
	   we're bailing because patching random memory addresses
	   in an unknown app_server is probably not a very
	   polite thing to do. */	
	if (init_window_colors() != B_OK)
	{
		/* If starting normally, alert them... */
		if (_startupAction == NORMAL_STARTUP_ACTION)
		{
			BAlert *buzzKill = new BAlert("",
										  CANNOT_BE_USED_MESSAGE,
										  "Bummer.");
			buzzKill->Go();
		}
		else	/* ...maybe they'll see this */
			printf("%s\n", CANNOT_BE_USED_MESSAGE);
		/* Abort */
		PostMessage(B_QUIT_REQUESTED);
	}
	else
	{	
		/* Load the settings file */
		LoadSettings();	
			
		switch (_startupAction)
		{
			/* Reset the colors to the defaults and update
			   the screen. */
			case RESET_TO_DEFAULTS_ACTION:
			{
				rgb_color colors[NUM_WINDOW_COLORS];
				get_window_colors(colors, DEFAULT_ONES);
				set_window_colors(colors, true);
				Quit();
			}
			break;

			/* Quietly update the colors to the user's
			   set colors (if any).  This is the option
			   people will want in their [User]Bootscripts. */
			case BOOTSCRIPT_ACTION:
			case USER_BOOTSCRIPT_ACTION:
			case UPDATE_WINDOW_COLORS_ACTION:
			{
				bool refresh = (_startupAction != BOOTSCRIPT_ACTION);
				BMessage *settings = Settings();
				rgb_color *color;
				ssize_t size;
				
				for (uint8 i = 0; i < NUM_WINDOW_COLORS; i++)
				{
					if (settings->FindData(PREFERRED_SET_ENTRY,
										   B_RGB_COLOR_TYPE,
										   i,
										   &color,
										   &size) == B_OK)
					{
						set_window_color((window_color)i,
										 *color,
										 (refresh && (i == 
										 	NUM_WINDOW_COLORS - 1)));
					}
				}
				Quit();
			}
			break;
				
			/* Pretty predictable... */
			case NORMAL_STARTUP_ACTION:
			default:
			{
				_window = new WindowColorsWindow();
				_window->Show();
			}
			break;
		}
	}
}


/* ...............................................................	
	WindowColorsApplication::Quit()
	Overridden to save the application settings before exiting.
   ............................................................... */

void WindowColorsApplication::Quit()
{
	SaveSettings();
	BApplication::Quit();
	shut_down_window_colors();
}


/* ...............................................................	
	WindowColorsApplication::SavePreferredSet()
	Saves the preferred window colors into the settings message,
	and writes the file to disk immediately.
   ............................................................... */

void WindowColorsApplication::SavePreferredSet(rgb_color *first)
{
	BMessage *settings = Settings();
	bool	 overwrite = settings->HasData(PREFERRED_SET_ENTRY,
									  	   B_RGB_COLOR_TYPE);
	
	for (uint8 i = 0; i < NUM_WINDOW_COLORS; i++)
	{
		if (overwrite)
			settings->ReplaceData(PREFERRED_SET_ENTRY,
								  B_RGB_COLOR_TYPE,
								  i,
								  first,
								  sizeof(rgb_color));
		else
			settings->AddData(PREFERRED_SET_ENTRY,
							  B_RGB_COLOR_TYPE,
							  first,
							  sizeof(rgb_color));
		first++;
	}
	SaveSettings();
}


/* ...............................................................	
	WindowColorsApplication::Settings()
	Returns the settings message, or creates a new one.
   ............................................................... */

BMessage *WindowColorsApplication::Settings()
{
	if (_settings == NULL)
		_settings = new BMessage(B_SIMPLE_DATA);
	return _settings;
}


/* ...............................................................	
	WindowColorsApplication::LoadSettings()
	Loads the application settings into the settings BMessage.
   ............................................................... */
  
void WindowColorsApplication::LoadSettings()
{	
	BPath path;
	find_directory(B_USER_SETTINGS_DIRECTORY, &path);
	path.Append(SETTINGS_FILE_NAME);
	
	BMessage *settings = Settings();
	BFile file = BFile(path.Path(), B_READ_ONLY);
	
	if (file.InitCheck() == B_OK)
		settings->Unflatten(&file);
}


/* ...............................................................	
	WindowColorsApplication::SaveSettings()
	Saves the application settings to a file.
   ............................................................... */

void WindowColorsApplication::SaveSettings()
{	
	BPath path;
	find_directory(B_USER_SETTINGS_DIRECTORY, &path);
	path.Append(SETTINGS_FILE_NAME);
	
	BMessage *settings = Settings();
	BFile file = BFile(path.Path(), 
					   B_READ_WRITE  |
					   B_CREATE_FILE | 
					   B_ERASE_FILE);

	if (file.InitCheck() == B_OK)
		settings->Flatten(&file);
}